using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.DAL.HL7AL
{

		#region Header

		//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		//<Developers>
		//	<Developer>Brian    lin</Developer>
		//</Developers>
		//<SiteName>Hines OIFO</SiteName>
		//<CreationDate>4/6/2004</CreationDate>
		//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		//<summary>This class provides support for Inbound HL7 messages coming from CPRS into VBECS.</summary>

		#endregion

	/// <summary>
	/// CprsOmgMessage
	/// </summary>
	public class CprsOmgMessage
	{
		private const string LAST_UPDATE_USER_PARAMETER_NAME = "@LastUpdateUser";
		private const string LAST_UPDATE_USER_PARAMETER_VALUE = "CPRS";
		private const string LAST_UPDATE_FUNSTEOR_VARAMETER_NAME = "@LastUpdateFunctionId";


		/// <summary>
		/// Blank Constructor for CprsOmgMessage
		/// </summary>
		public CprsOmgMessage() {}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7188"> 
		///		<ExpectedInput>Valid cprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>DataTable containing OrderedTest information</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7189"> 
		///		<ExpectedInput>Invalid cprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Given a CPRS order #, brings back an ordered test
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <returns><see cref="DataTable"/> with associated OrderedTests in VBECS.</returns>
		public static DataTable GetOrderedTestbyCPRSOrderNumber(string cprsOrderNumber)
		{
			DataSet ds;

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedTestbyCprsOrderNumber.CprsOrderNumber, System.Data.SqlDbType.VarChar)
			};
			prms[0].Value = cprsOrderNumber;
			
			try
			{
				ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedTestbyCprsOrderNumber.StoredProcName, prms);
			}
			catch (SqlException ex)
			{
				ExceptionManager.Publish(ex);
				throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.DataRetrieveError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString);
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.SystemRetrieveError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString);
			}

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7190"> 
		///		<ExpectedInput>Valid cprsOrderableItemId</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ComponentClass information</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7191"> 
		///		<ExpectedInput>Invalid cprsOrderableItemId</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///	Gets orderable test ID or component 
		///	class ID by CPRS orderable item ID
		///	</summary>
		///	<param name="cprsOrderableItemID">CPRS orderable item ID</param>
		///	<returns><see cref="DataTable"/> containing component class ID and orderable test ID </returns>
		public static DataTable GetOrderableTestIdOrComponentClassIdFromCprsOrderableItemId(int cprsOrderableItemID)
		{
			SqlParameter[] _prms =
			{
				new SqlParameter(STOREDPROC.GetOrderableTestIDOrComponentClassIDFromCPRSOrderableItemID.CprsOrderableItemId, System.Data.SqlDbType.Int)
			};

			_prms[0].Value = cprsOrderableItemID;
			
			try
			{
				return ( Common.StoredProcedure.GetData(STOREDPROC.GetOrderableTestIDOrComponentClassIDFromCPRSOrderableItemID.StoredProcName, _prms).Tables[0]);
			}
			catch (SqlException ex)
			{
				ExceptionManager.Publish(ex);
				throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.DataRetrieveError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString);
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.SystemRetrieveError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString);
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7192"> 
		///		<ExpectedInput>Valid cprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>DataTable containg OrderedComponentInformation</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7193"> 
		///		<ExpectedInput>Invalid cprsOrderNumber</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Given a CPRS order #, brings back an ordered component
		/// </summary>
		/// <param name="cprsOrderNumber">CPRS order number</param>
		/// <returns><see cref="DataTable"/> containing an ordered component data</returns>
		public static DataTable GetOrderedComponentbyCPRSOrderNumber(string cprsOrderNumber)
		{
			DataSet ds;

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedComponentbyCprsOrderNumber.CprsOrderNumber, System.Data.SqlDbType.VarChar)
			};
			prms[0].Value = cprsOrderNumber;
			
			try
			{
				ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedComponentbyCprsOrderNumber.StoredProcName, prms);
			}
			catch (SqlException ex)
			{
				ExceptionManager.Publish(ex);
				throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.DataRetrieveError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString);
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.SystemRetrieveError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString);
			}

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7194"> 
		///		<ExpectedInput>Valid orderGroupNumber</ExpectedInput>
		///		<ExpectedOutput>DataTable containing PatientOrder information</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7195"> 
		///		<ExpectedInput>Invalid orderGroupNumber</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Given a order group #, brings back a patient order.  Used by the CPRS HL7 interface.
		/// </summary>
		/// <param name="orderGroupNumber">Order group number</param>
		/// <returns><see cref="DataTable"/> containing patient order data</returns>
		public static DataTable GetPatientOrderbyOrderGroupNumber(string orderGroupNumber)
		{
			SqlParameter[] _prms =
			{
				new SqlParameter(STOREDPROC.HL7GetPatientOrderbyOrderGroupNumber.OrderGroupNumber, System.Data.SqlDbType.VarChar)
			};

			_prms[0].Value = orderGroupNumber;
			
			return ( Common.StoredProcedure.GetData(STOREDPROC.HL7GetPatientOrderbyOrderGroupNumber.StoredProcName, _prms).Tables[0]);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7196"> 
		///		<ExpectedInput>Valid patient ICN and vistaPatientID</ExpectedInput>
		///		<ExpectedOutput>DataTable containig patient information</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7197"> 
		///		<ExpectedInput>Invalid patinet ICN and vistaPatientID</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Searches for and returns patients matching input criteria from VBECS
		/// </summary>
		/// <param name="patientICN">Patient ICN</param>
		/// <param name="vistaPatientID">Vista Patient ID</param>
		/// <returns><see cref="DataTable"/> of patients matching search criteria</returns>
		public static DataTable GetHl7Patient(string patientICN, string vistaPatientID)
		{
			SqlParameter[] _prms =
			{
				new SqlParameter(STOREDPROC.HL7GetPatient.PatientICN, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.HL7GetPatient.VistaPatientId, System.Data.SqlDbType.VarChar),
			};
			_prms[0].Value = patientICN;
			_prms[1].Value = vistaPatientID;

			return Common.StoredProcedure.GetData(STOREDPROC.HL7GetPatient.StoredProcName, _prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7198"> 
		///		<ExpectedInput>Valid DataTable containing PatientTreatment data</ExpectedInput>
		///		<ExpectedOutput>DataTable of patients</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7199"> 
		///		<ExpectedInput>DataTable containing invalid PatientTreatment data</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Searches for and returns Patient Treatment matching input criteria from HL7 Message
		/// </summary>
		/// <param name="patientTreatment">PatientTreatment DataTable</param>
		/// <returns><see cref="DataTable"/> of patients matching search criteria</returns>
		public static DataTable GetHl7PatientTreatment(DataTable patientTreatment)
		{
			SqlParameter[] _prms =
			{
				new SqlParameter(STOREDPROC.HL7GetPatientTreatment.PatientGuid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.HL7GetPatientTreatment.PatientTreatingFacility, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.HL7GetPatientTreatment.PatientLocation, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.HL7GetPatientTreatment.PatientRoomBed, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.HL7GetPatientTreatment.InPatientIndicator, System.Data.SqlDbType.Bit),
				new SqlParameter(STOREDPROC.HL7GetPatientTreatment.PatientAdmittingDiagnosis, System.Data.SqlDbType.VarChar),
			};

			_prms[0].Value = patientTreatment.Rows[0]["PatientGuid"];
			_prms[1].Value = patientTreatment.Rows[0]["PatientTreatingFacility"];
			_prms[2].Value = patientTreatment.Rows[0]["PatientLocation"];
			_prms[3].Value = patientTreatment.Rows[0]["PatientRoomBed"];
			_prms[4].Value = patientTreatment.Rows[0]["InPatientIndicator"];
			_prms[5].Value = patientTreatment.Rows[0]["PatientAdmittingDiagnosis"];

			return Common.StoredProcedure.GetData(STOREDPROC.HL7GetPatientTreatment.StoredProcName, _prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7200"> 
		///		<ExpectedInput>Valid DataTable containing OrderedTest data</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="7202"> 
		///		<ExpectedInput>Valid DataTable containing OrderedComponent data</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7201"> 
		///		<ExpectedInput>DataTable containing invalid data</ExpectedInput>
		///		<ExpectedOutput>!=0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CancelPatientOrder
		/// </summary>
		/// <param name="dtTest"></param>
		/// <returns></returns>
		public int CancelPatientOrder(DataTable dtTest)
		{
			string sprocName;

			if (dtTest.Rows.Count > 0)
			{
				if (dtTest.Columns.Contains(TABLE.ComponentClass.ComponentClassId))	//Ordered component
				{
					sprocName = STOREDPROC.HL7UpdateOrderedComponent.StoredProcName;
				}

				else
					//Ordered test
				{
					sprocName = STOREDPROC.HL7UpdateOrderedTest.StoredProcName;
				}			

				return (new Common.StoredProcedure().TransactionalGetValue(sprocName, dtTest));
			}
			else
				return -1;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7203"> 
		///		<ExpectedInput>Valid DataTables containing OrderedTest information</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7205"> 
		///		<ExpectedInput>Valid DataTables containing OrderedComponent information</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7204"> 
		///		<ExpectedInput>Invalid DataTables</ExpectedInput>
		///		<ExpectedOutput>!=0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_88.10
		/// Saves patient, order, and (ordered test or component test) information 
		/// from CPRS (VISTA) into VBECS via the VBECS HL7 application.  Please do not remove.
		/// </summary>
		/// <param name="dtPatient">Patient data</param>
		/// <param name="dtPatientTreatment">Patient Treatment data</param>
		/// <param name="dtPatientOrder">Patient order data</param>
		/// <param name="dtTest">Test data</param>
		/// <returns>0 on success. Otherwise - non-zero value.</returns>
		public static int UpdateVistaPatientOrder (DataTable dtPatient, DataTable dtPatientTreatment, DataTable dtPatientOrder, DataTable dtTest)
		{
			string[] sprocNames;
			DataTable[] dtVistaDataTables;

			if (dtTest.Rows.Count > 0)
			{
				if (dtPatient.Rows.Count > 0)
				{
					sprocNames = new string[4];
					sprocNames.SetValue(STOREDPROC.HL7InsertPatient.StoredProcName, 0);
					sprocNames.SetValue(STOREDPROC.HL7InsertPatientTreatment.StoredProcName, 1);
					//BR_88.10
					sprocNames.SetValue(STOREDPROC.HL7UpdatePatientOrder.StoredProcName, 2);

					dtVistaDataTables = new DataTable[4];
					dtVistaDataTables. SetValue(dtPatient, 0);
					dtVistaDataTables. SetValue(dtPatientTreatment, 1);
					dtVistaDataTables. SetValue(dtPatientOrder, 2);
					dtVistaDataTables. SetValue(dtTest, 3);
				}
				else if (dtPatientTreatment.Rows.Count > 0)	//Division information exists.  Insert division data table.
				{		
					sprocNames = new string[3];
					sprocNames.SetValue(STOREDPROC.HL7InsertPatientTreatment.StoredProcName, 0);
					sprocNames.SetValue(STOREDPROC.HL7UpdatePatientOrder.StoredProcName, 1);

					dtVistaDataTables = new DataTable[3];
					dtVistaDataTables. SetValue(dtPatientTreatment, 0);
					dtVistaDataTables. SetValue(dtPatientOrder, 1);
					dtVistaDataTables. SetValue(dtTest, 2);
				}
			
				else if (dtPatientOrder.Rows.Count > 0)
				{
					sprocNames = new string[2];
					sprocNames.SetValue(STOREDPROC.HL7UpdatePatientOrder.StoredProcName, 0);
				
					dtVistaDataTables = new DataTable[2];
					dtVistaDataTables. SetValue(dtPatientOrder, 0);
					dtVistaDataTables. SetValue(dtTest, 1);

				}
				else if (dtTest.Columns.Contains(TABLE.ComponentClass.ComponentClassId))	//Ordered component
				{
					sprocNames =  new string[1];
					dtVistaDataTables = new DataTable[1];
					dtVistaDataTables.SetValue(dtTest, 0);
				}

				else
				{
					sprocNames =  new string[1];
					dtVistaDataTables = new DataTable[1];
					dtVistaDataTables.SetValue(dtTest, 0);
				}
				if (dtTest.Columns.Contains(TABLE.ComponentClass.ComponentClassId))	//Ordered component
				{
					sprocNames.SetValue(STOREDPROC.HL7UpdateOrderedComponent.StoredProcName, sprocNames.Length-1);
				}

				else
					//Ordered test
				{
					sprocNames.SetValue(STOREDPROC.HL7UpdateOrderedTest.StoredProcName, sprocNames.Length-1);
				}			

				return (new Common.StoredProcedure(Common.ConnectionType.VBECS).TransactionalGetValue(sprocNames, dtVistaDataTables));
			}
			else
				return -1;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7206"> 
		///		<ExpectedInput>Valid DataTable</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7207"> 
		///		<ExpectedInput>Invalid DataTable</ExpectedInput>
		///		<ExpectedOutput>!=0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates a Patient order with Lab test results from CRPS via the VBECS HL7 Application.
		/// Please do not remove.
		/// </summary>
		/// <param name="dtPatientTest">dtPatientTest</param>
		/// <returns>Integer</returns>
		public static int UpdatePatientPreviousResults(DataTable dtPatientTest)
		{
			if (dtPatientTest.Rows.Count > 0)
				return (new Common.StoredProcedure(Common.ConnectionType.VBECS)).TransactionalGetValue(STOREDPROC.HL7UpdatePatientTest.StoredProcName, dtPatientTest);
			else
				return -1;
		}

//		public static int CancelPatientOrder( int vbecsOrderNumber, string cancelReason )
//		{
//			return (
//		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7208"> 
		///		<ExpectedInput>Valid OrderType</ExpectedInput>
		///		<ExpectedOutput>Valid OrderTableType</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7209"> 
		///		<ExpectedInput>Invalid OrderType</ExpectedInput>
		///		<ExpectedOutput>ERROR</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Utitility to identify the Order Type based on the CprsOrderableItemId in the HL7 order message.
		/// </summary>
		/// <param name="OrderType">string representing the order type used in the HL7 message.</param>
		/// <returns>string value of "test" or "comp" depending on input.  May return "ERROR" if not a valid OrderType.</returns>
		public static string GetOrderTableTypeID(string OrderType)
		{
			
			switch (OrderType)
			{
				case "1":
				case "7":
				case "8":
				case "9":
				case "10":
					return "test";
				case "2":
				case "3":
				case "4":
				case "5":
				case "6":
				case "11":
					return "comp";
				default:
					return "ERROR";
			}
		}
	}
}
